<?php

namespace App\Http\Controllers\Api;

use App\Models\Inventory;
use App\Models\InventoryMovement;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class InventoryController extends Controller
{
    /**
     * Get inventory list with filters.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function index(Request $request): JsonResponse
    {
        $query = Inventory::query();

        // Filter by warehouse
        if ($request->has('warehouse_id')) {
            $query->where('warehouse_id', $request->warehouse_id);
        }

        // Filter by low stock
        if ($request->has('low_stock') && $request->low_stock === 'true') {
            $query->whereRaw('quantity <= reorder_level');
        }

        // Filter by product
        if ($request->has('product_id')) {
            $query->where('product_id', $request->product_id);
        }

        // Pagination
        $perPage = $request->get('per_page', 15);
        $inventories = $query->paginate($perPage);

        return response()->json([
            'success' => true,
            'message' => 'موجودی‌های دریافت‌شده',
            'data' => $inventories->items(),
            'pagination' => [
                'current_page' => $inventories->currentPage(),
                'total' => $inventories->total(),
                'per_page' => $inventories->perPage(),
                'last_page' => $inventories->lastPage(),
            ],
        ]);
    }

    /**
     * Get single inventory details.
     *
     * @param Inventory $inventory
     * @return JsonResponse
     */
    public function show(Inventory $inventory): JsonResponse
    {
        return response()->json([
            'success' => true,
            'message' => 'موجودی دریافت‌شده',
            'data' => $inventory,
        ]);
    }

    /**
     * Update inventory quantity.
     *
     * @param Request $request
     * @param Inventory $inventory
     * @return JsonResponse
     */
    public function update(Request $request, Inventory $inventory): JsonResponse
    {
        $validated = $request->validate([
            'quantity' => 'required|integer',
            'reorder_level' => 'sometimes|integer',
            'warehouse_id' => 'sometimes|exists:warehouses,id',
        ]);

        if (isset($validated['quantity'])) {
            $oldQuantity = $inventory->quantity;
            $quantity_change = $validated['quantity'] - $oldQuantity;
            
            $inventory->update(['quantity' => $validated['quantity']]);

            // Log the movement
            InventoryMovement::create([
                'product_id' => $inventory->product_id,
                'warehouse_id' => $inventory->warehouse_id,
                'movement_type' => $quantity_change > 0 ? 'adjustment' : 'adjustment',
                'quantity' => $quantity_change,
                'notes' => 'تعدیل موجودی',
                'user_id' => auth()->id(),
            ]);
        }

        if (isset($validated['reorder_level'])) {
            $inventory->update(['reorder_level' => $validated['reorder_level']]);
        }

        return response()->json([
            'success' => true,
            'message' => 'موجودی بروزرسانی شد',
            'data' => $inventory->fresh(),
        ]);
    }

    /**
     * Get inventory movements/history.
     *
     * @param Request $request
     * @param Inventory $inventory
     * @return JsonResponse
     */
    public function movements(Request $request, Inventory $inventory): JsonResponse
    {
        $query = InventoryMovement::where('product_id', $inventory->product_id)
            ->where('warehouse_id', $inventory->warehouse_id);

        // Filter by type
        if ($request->has('type')) {
            $query->where('movement_type', $request->type);
        }

        // Date range
        if ($request->has('start_date') && $request->has('end_date')) {
            $query->whereBetween('created_at', [
                $request->start_date,
                $request->end_date,
            ]);
        }

        $movements = $query->orderBy('created_at', 'desc')->paginate(20);

        return response()->json([
            'success' => true,
            'message' => 'تاریخچه موجودی دریافت‌شده',
            'data' => $movements->items(),
            'pagination' => [
                'current_page' => $movements->currentPage(),
                'total' => $movements->total(),
                'per_page' => $movements->perPage(),
                'last_page' => $movements->lastPage(),
            ],
        ]);
    }

    /**
     * Get low stock items.
     *
     * @return JsonResponse
     */
    public function lowStock(): JsonResponse
    {
        $lowStockItems = Inventory::whereRaw('quantity <= reorder_level')
            ->paginate(20);

        return response()->json([
            'success' => true,
            'message' => 'آیتم‌های موجودی پایین دریافت‌شده',
            'data' => $lowStockItems->items(),
            'count' => Inventory::whereRaw('quantity <= reorder_level')->count(),
        ]);
    }

    /**
     * Get inventory summary by warehouse.
     *
     * @return JsonResponse
     */
    public function summary(): JsonResponse
    {
        $summary = Inventory::selectRaw('warehouse_id, COUNT(*) as item_count, SUM(quantity) as total_quantity')
            ->groupBy('warehouse_id')
            ->with('warehouse')
            ->get();

        return response()->json([
            'success' => true,
            'message' => 'خلاصه موجودی',
            'data' => $summary,
        ]);
    }

    /**
     * Sync inventory with main store.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function sync(Request $request): JsonResponse
    {
        // This will be implemented to sync with the main SeoShop database
        // Implementation depends on the actual database connection and sync strategy
        
        try {
            // Sync logic here
            return response()->json([
                'success' => true,
                'message' => 'موجودی با موفقیت هماهنگ شد',
                'synced_records' => 0,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'خطا در هماهنگی: ' . $e->getMessage(),
            ], 500);
        }
    }
}
