<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class ExpirationAlert extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'expiration_alerts';

    /**
     * Indicates if the model should be timestamped.
     *
     * @var bool
     */
    public $timestamps = false;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'batch_id',
        'alert_level',
        'days_remaining',
        'quantity',
        'alert_date',
        'sent_to',
        'acknowledged',
        'action_taken',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'days_remaining' => 'integer',
        'quantity' => 'integer',
        'alert_date' => 'datetime',
        'acknowledged' => 'boolean',
    ];

    /**
     * Alert levels with Persian labels.
     *
     * @var array
     */
    public const LEVELS = [
        'warning' => 'هشدار',
        'critical' => 'بحرانی',
        'expired' => 'منقضی‌شده',
    ];

    /**
     * Get the batch for the alert.
     */
    public function batch()
    {
        return $this->belongsTo(ProductBatch::class, 'batch_id', 'id');
    }

    /**
     * Get the alert level label in Persian.
     */
    public function getLevelLabel()
    {
        return self::LEVELS[$this->alert_level] ?? $this->alert_level;
    }

    /**
     * Mark alert as acknowledged.
     */
    public function acknowledge($userId = null)
    {
        $this->acknowledged = true;
        $this->save();
        
        return $this;
    }

    /**
     * Record action taken on alert.
     */
    public function recordAction($action, $userId = null)
    {
        $this->action_taken = $action;
        $this->acknowledged = true;
        $this->save();
        
        return $this;
    }

    /**
     * Scope: get unacknowledged alerts.
     */
    public function scopeUnacknowledged($query)
    {
        return $query->where('acknowledged', false);
    }

    /**
     * Scope: get alerts by level.
     */
    public function scopeByLevel($query, $level)
    {
        return $query->where('alert_level', $level);
    }

    /**
     * Scope: get critical alerts.
     */
    public function scopeCritical($query)
    {
        return $query->where('alert_level', 'critical')
            ->orWhere('alert_level', 'expired');
    }
}
