<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Inventory extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'inventories';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'product_id',
        'warehouse_id',
        'quantity',
        'reorder_level',
        'last_stock_check',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'quantity' => 'integer',
        'reorder_level' => 'integer',
        'last_stock_check' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Get the warehouse for the inventory.
     */
    public function warehouse()
    {
        return $this->belongsTo(Warehouse::class, 'warehouse_id', 'id');
    }

    /**
     * Get the movements for the inventory.
     */
    public function movements()
    {
        return $this->hasMany(InventoryMovement::class, 'product_id', 'product_id')
            ->where('warehouse_id', $this->warehouse_id);
    }

    /**
     * Check if inventory is below reorder level.
     */
    public function isLowStock()
    {
        return $this->quantity <= $this->reorder_level;
    }

    /**
     * Update inventory quantity.
     */
    public function updateQuantity($quantity, $reason = null, $userId = null)
    {
        $oldQuantity = $this->quantity;
        $this->quantity += $quantity;
        $this->save();

        // Log the movement
        InventoryMovement::create([
            'product_id' => $this->product_id,
            'warehouse_id' => $this->warehouse_id,
            'movement_type' => 'adjustment',
            'quantity' => $quantity,
            'notes' => $reason,
            'user_id' => $userId,
        ]);

        return $this;
    }
}
