<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class InventoryMovement extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'inventory_movements';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'product_id',
        'warehouse_id',
        'movement_type',
        'quantity',
        'reference_id',
        'reference_type',
        'notes',
        'user_id',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'quantity' => 'integer',
        'created_at' => 'datetime',
    ];

    /**
     * The attributes that contain movement types.
     *
     * @var array
     */
    public const TYPES = [
        'purchase' => 'خرید',
        'sale' => 'فروش',
        'adjustment' => 'تعدیل',
        'transfer_out' => 'انتقال از انبار',
        'transfer_in' => 'دریافت انتقال',
        'damage' => 'آسیب',
        'return' => 'بازگشت',
        'count' => 'موجودی',
        'waste' => 'ضایعات',
        'other' => 'سایر',
    ];

    /**
     * Scope: get movements by type.
     */
    public function scopeByType($query, $type)
    {
        return $query->where('movement_type', $type);
    }

    /**
     * Scope: get movements by date range.
     */
    public function scopeDateBetween($query, $startDate, $endDate)
    {
        return $query->whereBetween('created_at', [$startDate, $endDate]);
    }

    /**
     * Get the warehouse for the movement.
     */
    public function warehouse()
    {
        return $this->belongsTo(Warehouse::class, 'warehouse_id', 'id');
    }

    /**
     * Get the user who made the movement.
     */
    public function user()
    {
        return $this->belongsTo(User::class, 'user_id', 'id');
    }

    /**
     * Get the movement type label in Persian.
     */
    public function getTypeLabel()
    {
        return self::TYPES[$this->movement_type] ?? $this->movement_type;
    }
}
