<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class InventoryTransfer extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'inventory_transfers';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'product_id',
        'source_warehouse_id',
        'destination_warehouse_id',
        'quantity',
        'status',
        'shipped_date',
        'received_date',
        'created_by',
        'received_by',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'quantity' => 'integer',
        'shipped_date' => 'datetime',
        'received_date' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Transfer statuses.
     *
     * @var array
     */
    public const STATUSES = [
        'pending' => 'در انتظار',
        'in_transit' => 'در حال انتقال',
        'received' => 'دریافت‌شده',
        'cancelled' => 'لغو‌شده',
    ];

    /**
     * Get the source warehouse.
     */
    public function sourceWarehouse()
    {
        return $this->belongsTo(Warehouse::class, 'source_warehouse_id', 'id');
    }

    /**
     * Get the destination warehouse.
     */
    public function destinationWarehouse()
    {
        return $this->belongsTo(Warehouse::class, 'destination_warehouse_id', 'id');
    }

    /**
     * Get the user who created the transfer.
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by', 'id');
    }

    /**
     * Get the user who received the transfer.
     */
    public function receiver()
    {
        return $this->belongsTo(User::class, 'received_by', 'id');
    }

    /**
     * Mark transfer as shipped.
     */
    public function ship($userId = null)
    {
        $this->status = 'in_transit';
        $this->shipped_date = now();
        $this->save();
        
        return $this;
    }

    /**
     * Mark transfer as received.
     */
    public function receive($userId = null)
    {
        $this->status = 'received';
        $this->received_date = now();
        $this->received_by = $userId;
        $this->save();
        
        // Update destination warehouse inventory
        $inventory = Inventory::where('product_id', $this->product_id)
            ->where('warehouse_id', $this->destination_warehouse_id)
            ->firstOrCreate(
                ['product_id' => $this->product_id, 'warehouse_id' => $this->destination_warehouse_id],
                ['quantity' => 0]
            );
        
        $inventory->updateQuantity($this->quantity, 'Transfer received', $userId);
        
        return $this;
    }

    /**
     * Cancel the transfer.
     */
    public function cancel()
    {
        $this->status = 'cancelled';
        $this->save();
        
        return $this;
    }

    /**
     * Get transit duration in hours.
     */
    public function getTransitDuration()
    {
        if (!$this->shipped_date || !$this->received_date) return null;
        return $this->shipped_date->diffInHours($this->received_date);
    }

    /**
     * Get status label in Persian.
     */
    public function getStatusLabel()
    {
        return self::STATUSES[$this->status] ?? $this->status;
    }

    /**
     * Scope: get pending transfers.
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * Scope: get in-transit transfers.
     */
    public function scopeInTransit($query)
    {
        return $query->where('status', 'in_transit');
    }

    /**
     * Scope: get received transfers.
     */
    public function scopeReceived($query)
    {
        return $query->where('status', 'received');
    }
}
