<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Carbon\Carbon;

class ProductBatch extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'product_batches';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'product_id',
        'batch_number',
        'warehouse_id',
        'quantity',
        'manufacturing_date',
        'expiration_date',
        'status',
        'alert_sent',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'quantity' => 'integer',
        'manufacturing_date' => 'date',
        'expiration_date' => 'date',
        'alert_sent' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * The attributes that contain status types.
     *
     * @var array
     */
    public const STATUSES = [
        'active' => 'فعال',
        'expired' => 'منقضی‌شده',
        'recalled' => 'فراخوان‌شده',
        'disposed' => 'دفع‌شده',
    ];

    /**
     * Get the warehouse for the batch.
     */
    public function warehouse()
    {
        return $this->belongsTo(Warehouse::class, 'warehouse_id', 'id');
    }

    /**
     * Get the alerts for the batch.
     */
    public function alerts()
    {
        return $this->hasMany(ExpirationAlert::class, 'batch_id', 'id');
    }

    /**
     * Get the movements for the batch.
     */
    public function movements()
    {
        return $this->hasMany(BatchMovement::class, 'batch_id', 'id');
    }

    /**
     * Calculate days until expiration.
     */
    public function getDaysToExpiry()
    {
        return $this->expiration_date->diffInDays(Carbon::today(), false);
    }

    /**
     * Check if batch is expired.
     */
    public function isExpired()
    {
        return $this->expiration_date->isPast();
    }

    /**
     * Check if batch needs warning alert.
     */
    public function needsWarning($warningDays = 30)
    {
        $daysToExpiry = $this->getDaysToExpiry();
        return $daysToExpiry <= $warningDays && $daysToExpiry > 0;
    }

    /**
     * Check if batch is critical.
     */
    public function isCritical($criticalDays = 7)
    {
        $daysToExpiry = $this->getDaysToExpiry();
        return $daysToExpiry <= $criticalDays && $daysToExpiry > 0;
    }

    /**
     * Scope: get active batches.
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    /**
     * Scope: get expired batches.
     */
    public function scopeExpired($query)
    {
        return $query->where('status', 'expired')
            ->orWhereDate('expiration_date', '<', Carbon::today());
    }

    /**
     * Scope: get batches expiring soon.
     */
    public function scopeExpiringsoon($query, $days = 30)
    {
        $startDate = Carbon::today();
        $endDate = Carbon::today()->addDays($days);
        
        return $query->active()
            ->whereBetween('expiration_date', [$startDate, $endDate]);
    }

    /**
     * Get the status label in Persian.
     */
    public function getStatusLabel()
    {
        return self::STATUSES[$this->status] ?? $this->status;
    }
}
