<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class ProductPrice extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'product_prices';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'product_id',
        'base_price',
        'selling_price',
        'wholesale_price',
        'discount_percentage',
        'discount_start',
        'discount_end',
        'currency',
        'effective_date',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'base_price' => 'decimal:0',
        'selling_price' => 'decimal:0',
        'wholesale_price' => 'decimal:0',
        'discount_percentage' => 'decimal:2',
        'discount_start' => 'datetime',
        'discount_end' => 'datetime',
        'effective_date' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Get profit margin percentage.
     */
    public function getProfitMargin()
    {
        if ($this->base_price == 0) return 0;
        return (($this->selling_price - $this->base_price) / $this->base_price) * 100;
    }

    /**
     * Get effective selling price (considering discount).
     */
    public function getEffectivePrice()
    {
        $now = now();
        $hasActiveDiscount = $this->discount_start && $this->discount_end 
            && $now->between($this->discount_start, $this->discount_end);
        
        if ($hasActiveDiscount && $this->discount_percentage > 0) {
            return $this->selling_price * (1 - $this->discount_percentage / 100);
        }
        
        return $this->selling_price;
    }

    /**
     * Scope: get current prices.
     */
    public function scopeCurrent($query)
    {
        return $query->where('effective_date', '<=', now())
            ->orderBy('effective_date', 'desc')
            ->first();
    }

    /**
     * Scope: get prices by product.
     */
    public function scopeByProduct($query, $productId)
    {
        return $query->where('product_id', $productId);
    }

    /**
     * Scope: get prices with active discounts.
     */
    public function scopeWithActiveDiscount($query)
    {
        $now = now();
        return $query->where('discount_percentage', '>', 0)
            ->where('discount_start', '<=', $now)
            ->where('discount_end', '>=', $now);
    }

    /**
     * Get the audit log for the price.
     */
    public function auditLogs()
    {
        return $this->hasMany(PriceAuditLog::class, 'product_id', 'product_id')
            ->orderBy('change_date', 'desc');
    }
}
