<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class StockCountItem extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'stock_count_items';

    /**
     * Indicates if the model should be timestamped.
     *
     * @var bool
     */
    public $timestamps = false;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'count_id',
        'product_id',
        'batch_id',
        'system_quantity',
        'physical_quantity',
        'notes',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'system_quantity' => 'integer',
        'physical_quantity' => 'integer',
    ];

    /**
     * Get the stock count.
     */
    public function stockCount()
    {
        return $this->belongsTo(StockCount::class, 'count_id', 'id');
    }

    /**
     * Get the batch.
     */
    public function batch()
    {
        return $this->belongsTo(ProductBatch::class, 'batch_id', 'id');
    }

    /**
     * Get variance (physical - system).
     */
    public function getVariance()
    {
        return $this->physical_quantity - $this->system_quantity;
    }

    /**
     * Check if item has discrepancy.
     */
    public function hasDiscrepancy()
    {
        return $this->getVariance() != 0;
    }

    /**
     * Get variance percentage.
     */
    public function getVariancePercentage()
    {
        if ($this->system_quantity == 0) {
            return $this->physical_quantity > 0 ? 100 : 0;
        }
        return abs($this->getVariance() / $this->system_quantity) * 100;
    }
}
