<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use Tymon\JWTAuth\Contracts\JWTSubject;

class User extends Authenticatable implements JWTSubject
{
    use HasApiTokens, HasFactory, Notifiable;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'users';

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'email',
        'phone',
        'role',
        'warehouse_id',
        'password',
        'status',
        'two_factor_enabled',
        'two_factor_secret',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'two_factor_secret',
        'remember_token',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
        'two_factor_enabled' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * User roles with permissions.
     *
     * @var array
     */
    public const ROLES = [
        'admin' => ['all'],
        'manager' => ['inventory.manage', 'price.manage', 'reports.view', 'users.manage'],
        'warehouse_staff' => ['inventory.view', 'inventory.update', 'batches.view'],
        'accountant' => ['reports.view', 'inventory.view', 'price.view', 'sync.view'],
        'viewer' => ['inventory.view', 'reports.view'],
    ];

    /**
     * Get the identifier that will be stored in the subject claim of the JWT.
     *
     * @return mixed
     */
    public function getJWTIdentifier()
    {
        return $this->getKey();
    }

    /**
     * Return a key value array, containing any custom claims to be added to the JWT.
     *
     * @return array
     */
    public function getJWTCustomClaims()
    {
        return [
            'role' => $this->role,
            'email' => $this->email,
            'warehouse_id' => $this->warehouse_id,
        ];
    }

    /**
     * Get the warehouse for the user.
     */
    public function warehouse()
    {
        return $this->belongsTo(Warehouse::class, 'warehouse_id', 'id');
    }

    /**
     * Get the activity logs for the user.
     */
    public function activityLogs()
    {
        return $this->hasMany(ActivityLog::class, 'user_id', 'id');
    }

    /**
     * Check if user has permission.
     */
    public function hasPermission($permission)
    {
        $rolePermissions = self::ROLES[$this->role] ?? [];
        return in_array('all', $rolePermissions) || in_array($permission, $rolePermissions);
    }

    /**
     * Check if user has role.
     */
    public function hasRole($role)
    {
        return $this->role === $role;
    }

    /**
     * Check if user is admin.
     */
    public function isAdmin()
    {
        return $this->role === 'admin';
    }

    /**
     * Check if user is active.
     */
    public function isActive()
    {
        return $this->status === 'active';
    }

    /**
     * Enable two-factor authentication.
     */
    public function enableTwoFactor($secret = null)
    {
        $this->two_factor_enabled = true;
        $this->two_factor_secret = $secret ?? \Illuminate\Support\Str::random(32);
        $this->save();
        
        return $this->two_factor_secret;
    }

    /**
     * Disable two-factor authentication.
     */
    public function disableTwoFactor()
    {
        $this->two_factor_enabled = false;
        $this->two_factor_secret = null;
        $this->save();
        
        return $this;
    }

    /**
     * Scope: get active users.
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    /**
     * Scope: get users by role.
     */
    public function scopeByRole($query, $role)
    {
        return $query->where('role', $role);
    }

    /**
     * Scope: get users by warehouse.
     */
    public function scopeByWarehouse($query, $warehouseId)
    {
        return $query->where('warehouse_id', $warehouseId);
    }
}
