<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Warehouse extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'warehouses';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'name',
        'location',
        'capacity',
        'manager_name',
        'phone',
        'status',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'capacity' => 'integer',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Get the inventories for the warehouse.
     */
    public function inventories()
    {
        return $this->hasMany(Inventory::class, 'warehouse_id', 'id');
    }

    /**
     * Get the batches for the warehouse.
     */
    public function batches()
    {
        return $this->hasMany(ProductBatch::class, 'warehouse_id', 'id');
    }

    /**
     * Get the stock counts for the warehouse.
     */
    public function stockCounts()
    {
        return $this->hasMany(StockCount::class, 'warehouse_id', 'id');
    }

    /**
     * Get the total inventory count for the warehouse.
     */
    public function getTotalInventory()
    {
        return $this->inventories()->sum('quantity');
    }

    /**
     * Get warehouse capacity utilization percentage.
     */
    public function getCapacityUtilization()
    {
        $totalInventory = $this->getTotalInventory();
        return $this->capacity > 0 ? round(($totalInventory / $this->capacity) * 100, 2) : 0;
    }
}
