<?php

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;
use App\Http\Controllers\Api\InventoryController;
use App\Http\Controllers\Api\PriceController;
use App\Http\Controllers\Api\ExpirationController;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
|
| Here is where you can register API routes for your application. These
| routes are loaded by the RouteServiceProvider within a group which
| is assigned the "api" middleware group. Enjoy building your API!
|
*/

Route::prefix('v1')->group(function () {
    // Public routes
    Route::post('/auth/login', 'Api\AuthController@login');
    Route::post('/auth/register', 'Api\AuthController@register');

    // Protected routes
    Route::middleware('auth:api')->group(function () {
        // Authentication routes
        Route::post('/auth/logout', 'Api\AuthController@logout');
        Route::get('/auth/me', 'Api\AuthController@me');
        Route::post('/auth/refresh', 'Api\AuthController@refresh');

        // Inventory Management Routes
        Route::prefix('inventory')->group(function () {
            Route::get('/', [InventoryController::class, 'index']);
            Route::get('/summary', [InventoryController::class, 'summary']);
            Route::get('/low-stock', [InventoryController::class, 'lowStock']);
            Route::get('/{inventory}', [InventoryController::class, 'show']);
            Route::put('/{inventory}', [InventoryController::class, 'update']);
            Route::get('/{inventory}/movements', [InventoryController::class, 'movements']);
            Route::post('/sync', [InventoryController::class, 'sync']);
        });

        // Price Management Routes
        Route::prefix('prices')->group(function () {
            Route::get('/', [PriceController::class, 'index']);
            Route::post('/', [PriceController::class, 'store']);
            Route::get('/analytics', [PriceController::class, 'analytics']);
            Route::get('/{price}', [PriceController::class, 'show']);
            Route::put('/{price}', [PriceController::class, 'update']);
            Route::post('/{price}/discount', [PriceController::class, 'applyDiscount']);
            Route::delete('/{price}/discount', [PriceController::class, 'removeDiscount']);
            Route::get('/product/{productId}/history', [PriceController::class, 'history']);
            Route::post('/sync', [PriceController::class, 'sync']);
        });

        // Expiration Management Routes
        Route::prefix('batches')->group(function () {
            Route::get('/', [ExpirationController::class, 'index']);
            Route::post('/', [ExpirationController::class, 'store']);
            Route::get('/expired', [ExpirationController::class, 'expired']);
            Route::get('/expiring-soon', [ExpirationController::class, 'expiringSoon']);
            Route::get('/alerts', [ExpirationController::class, 'alerts']);
            Route::post('/alerts', [ExpirationController::class, 'createAlert']);
            Route::put('/alerts/{alert}/acknowledge', [ExpirationController::class, 'acknowledgeAlert']);
            Route::post('/alerts/{alert}/action', [ExpirationController::class, 'recordAction']);
            Route::get('/report', [ExpirationController::class, 'report']);
            Route::get('/{batch}', [ExpirationController::class, 'show']);
            Route::put('/{batch}', [ExpirationController::class, 'update']);
        });

        // Warehouse Management Routes
        Route::prefix('warehouses')->group(function () {
            Route::get('/', 'Api\WarehouseController@index');
            Route::post('/', 'Api\WarehouseController@store');
            Route::get('/{warehouse}', 'Api\WarehouseController@show');
            Route::put('/{warehouse}', 'Api\WarehouseController@update');
            Route::delete('/{warehouse}', 'Api\WarehouseController@destroy');
        });

        // Stock Count Routes
        Route::prefix('stock-counts')->group(function () {
            Route::get('/', 'Api\StockCountController@index');
            Route::post('/', 'Api\StockCountController@store');
            Route::get('/{stockCount}', 'Api\StockCountController@show');
            Route::put('/{stockCount}', 'Api\StockCountController@update');
            Route::post('/{stockCount}/complete', 'Api\StockCountController@complete');
            Route::get('/{stockCount}/items', 'Api\StockCountController@items');
            Route::post('/{stockCount}/items', 'Api\StockCountController@addItem');
        });

        // Inventory Transfer Routes
        Route::prefix('transfers')->group(function () {
            Route::get('/', 'Api\TransferController@index');
            Route::post('/', 'Api\TransferController@store');
            Route::get('/{transfer}', 'Api\TransferController@show');
            Route::post('/{transfer}/ship', 'Api\TransferController@ship');
            Route::post('/{transfer}/receive', 'Api\TransferController@receive');
            Route::post('/{transfer}/cancel', 'Api\TransferController@cancel');
        });

        // Reports Routes
        Route::prefix('reports')->group(function () {
            Route::get('/inventory-summary', 'Api\ReportController@inventorySummary');
            Route::get('/low-stock-report', 'Api\ReportController@lowStockReport');
            Route::get('/price-changes', 'Api\ReportController@priceChanges');
            Route::get('/expiration-report', 'Api\ReportController@expirationReport');
            Route::get('/warehouse-performance', 'Api\ReportController@warehousePerformance');
            Route::get('/sync-status', 'Api\ReportController@syncStatus');
        });

        // User Management Routes
        Route::prefix('users')->group(function () {
            Route::get('/', 'Api\UserController@index');
            Route::post('/', 'Api\UserController@store');
            Route::get('/{user}', 'Api\UserController@show');
            Route::put('/{user}', 'Api\UserController@update');
            Route::delete('/{user}', 'Api\UserController@destroy');
            Route::post('/{user}/roles', 'Api\UserController@updateRole');
            Route::post('/{user}/2fa/enable', 'Api\UserController@enable2FA');
            Route::post('/{user}/2fa/disable', 'Api\UserController@disable2FA');
        });

        // System Settings Routes
        Route::prefix('settings')->group(function () {
            Route::get('/', 'Api\SettingController@index');
            Route::get('/{key}', 'Api\SettingController@show');
            Route::put('/{key}', 'Api\SettingController@update');
        });

        // Activity Logs Routes
        Route::get('/activity-logs', 'Api\ActivityLogController@index');
        Route::get('/activity-logs/{log}', 'Api\ActivityLogController@show');

        // Sync Routes
        Route::post('/sync/run', 'Api\SyncController@run');
        Route::get('/sync/status', 'Api\SyncController@status');
        Route::get('/sync/logs', 'Api\SyncController@logs');
    });
});

// Health check endpoint
Route::get('/health', function () {
    return response()->json([
        'status' => 'ok',
        'message' => 'سیستم آماده است',
        'timestamp' => now()->toIso8601String(),
    ]);
});
